/*
 * Decompiled with CFR 0.152.
 */
package com.couchbase.connector.cluster.consul;

import com.couchbase.connector.cluster.consul.ConsulDocumentWatcher;
import com.couchbase.connector.cluster.consul.ConsulHelper;
import com.couchbase.connector.cluster.consul.DocumentKeys;
import com.couchbase.connector.cluster.consul.ReactorHelper;
import com.couchbase.connector.cluster.consul.rpc.RpcEndpoint;
import com.couchbase.connector.config.ConfigException;
import com.couchbase.connector.config.common.ConsulConfig;
import com.couchbase.consul.ConsulOps;
import com.couchbase.consul.ConsulResponse;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.google.common.base.Strings;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableSet;
import com.google.common.net.HostAndPort;
import com.google.common.util.concurrent.Uninterruptibles;
import java.io.Closeable;
import java.io.IOException;
import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import org.jspecify.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Flux;

public class ConsulContext
implements Closeable {
    private static final Logger log = LoggerFactory.getLogger(ConsulContext.class);
    private final DocumentKeys keys;
    private final String serviceName;
    private final String serviceId;
    private final String serviceUuid = UUID.randomUUID().toString();
    private final ConsulDocumentWatcher documentWatcher;
    private final ConsulOps client;
    private final String ttlCheckName;
    private final ConsulConfig config;
    private final HostAndPort address;

    public ConsulContext(HostAndPort address, ConsulConfig consulConfig, String serviceName, @Nullable String serviceIdOrNull) {
        this.config = Objects.requireNonNull(consulConfig);
        this.serviceName = Objects.requireNonNull(serviceName);
        this.serviceId = Optional.ofNullable(serviceIdOrNull).orElse(serviceName);
        this.client = new ConsulOps(address, consulConfig);
        this.ttlCheckName = "service:" + this.serviceId;
        this.documentWatcher = new ConsulDocumentWatcher(this.client);
        this.keys = new DocumentKeys(this.client, this.documentWatcher, serviceName);
        this.address = Objects.requireNonNull(address);
    }

    public boolean pause() throws IOException, TimeoutException {
        return this.keys.pause();
    }

    public void resume() throws IOException {
        this.keys.resume();
    }

    public ConsulOps client() {
        return this.client;
    }

    public DocumentKeys keys() {
        return this.keys;
    }

    public List<RpcEndpoint> rpcEndpoints() {
        return this.keys.listRpcEndpoints();
    }

    public String readConfig() {
        log.info("Reading connector config from Consul key: {}", (Object)this.keys.config());
        return ReactorHelper.blockSingle(this.client.kv().readOneKeyAsString(this.keys.config())).orElseThrow(() -> new ConfigException("Connector config does not exist in Consul. Missing KV key: " + this.keys.config()));
    }

    public <T> T readConfigOrExit(Function<String, T> configParser) {
        try {
            String configString = this.readConfig();
            if (Strings.isNullOrEmpty((String)configString)) {
                System.err.println("ERROR: Connector configuration document does not exist, or is empty.");
                System.exit(1);
            }
            try {
                return configParser.apply(configString);
            }
            catch (Exception e) {
                System.err.println("ERROR: Connector configuration document is malformed.");
                e.printStackTrace();
                System.exit(1);
                throw new AssertionError((Object)"unreachable");
            }
        }
        catch (Exception e) {
            System.err.println("ERROR: Failed to read connector configuration document.");
            e.printStackTrace();
            System.exit(1);
            throw new AssertionError((Object)"unreachable");
        }
    }

    public String serviceName() {
        return this.serviceName;
    }

    public String serviceId() {
        return this.serviceId;
    }

    public void register(Duration healthCheckTtl) {
        this.client.agent().register(Map.of("Id", this.serviceId, "Name", this.serviceName, "Meta", Map.of("uuid", this.serviceUuid), "Tags", List.of("couchbase-elasticsearch-connector"), "Check", Map.of("CheckID", this.ttlCheckName, "TTL", healthCheckTtl.toSeconds() + "s", "DeregisterCriticalServiceAfter", this.config.deregisterCriticalServiceAfter().toSeconds() + "s")), Map.of("replace-existing-checks", "")).block();
    }

    public String createSession() {
        ConsulResponse<String> sessionResponse = ReactorHelper.blockSingle(this.client.session().createSession(Map.of("Name", "couchbase:cbes:" + this.serviceId, "Behavior", "delete", "LockDelay", "15s", "Checks", List.of("serfHealth", "service:" + this.serviceId)), Map.of()));
        return sessionResponse.body();
    }

    public void passHealthCheck() {
        this.client.agent().check("pass", this.ttlCheckName, Map.of("note", "(" + this.serviceId + ") OK")).block();
    }

    private void failHealthCheck(String message) {
        this.client.agent().check("fail", this.ttlCheckName, Map.of("note", "(" + this.serviceId + ") " + message)).block();
    }

    private void deregister() {
        this.client.agent().deregister(this.serviceId).block();
    }

    public Flux<Optional<String>> watchConfig() {
        return this.documentWatcher.watch(this.keys().config());
    }

    public Flux<Optional<String>> watchControl() {
        return this.documentWatcher.watch(this.keys().control());
    }

    public ConsulDocumentWatcher documentWatcher() {
        return this.documentWatcher;
    }

    public Flux<ImmutableSet<String>> watchServiceHealth(Duration quietPeriod) {
        return ConsulHelper.watchServiceHealth(this.client, this.serviceName, quietPeriod);
    }

    @Override
    public void close() {
        this.client.close();
    }

    public void runCleanup(Runnable cleanupTask) {
        AtomicReference deferred = new AtomicReference();
        Thread thread = new Thread(() -> {
            try {
                cleanupTask.run();
            }
            catch (Throwable t) {
                deferred.set(t);
            }
        });
        thread.start();
        int timeoutSeconds = 30;
        Uninterruptibles.joinUninterruptibly((Thread)thread, (long)30L, (TimeUnit)TimeUnit.SECONDS);
        if (thread.isAlive()) {
            throw new IllegalStateException("cleanup thread failed to complete within 30 seconds.");
        }
        Throwable t = (Throwable)deferred.get();
        if (t != null) {
            Throwables.throwIfUnchecked((Throwable)t);
            throw new RuntimeException(t);
        }
    }

    public boolean acquireLock(String key, String value, String sessionId) {
        return ReactorHelper.blockSingle(this.client.kv().upsertKey(key, value, Map.of("acquire", sessionId))).body();
    }

    public boolean unlockAndDelete(String key, String sessionId) {
        return ConsulHelper.unlockAndDelete(this.client.httpClient(), key, sessionId);
    }

    public String myEndpointId() {
        ObjectNode self = ReactorHelper.blockSingle(this.client.agent().self()).body();
        return ConsulContext.formatEndpointId(Objects.requireNonNull(self.path("Member").path("Name").textValue(), "missing Member.Name"), Objects.requireNonNull(self.path("Member").path("Addr").textValue(), "missing Member.Addr"), this.serviceId);
    }

    public static String formatEndpointId(String nodeName, String nodeAddress, String serviceId) {
        return String.join((CharSequence)"::", nodeName, nodeAddress, serviceId);
    }

    public void reportShutdown(@Nullable Throwable cause) {
        try {
            if (cause != null) {
                this.failHealthCheck("Shutdown triggered by exception: " + Throwables.getStackTraceAsString((Throwable)cause));
                return;
            }
            if (this.config.deregisterServiceOnGracefulShutdown()) {
                this.deregister();
                return;
            }
            this.failHealthCheck("Graceful shutdown complete.");
        }
        catch (Throwable t) {
            System.err.println("Failed to report termination to Consul agent.");
            if (cause != null) {
                t.addSuppressed(cause);
            }
            t.printStackTrace();
        }
    }

    public String consulAddress() {
        return this.address.toString();
    }
}

