/*
 * Decompiled with CFR 0.152.
 */
package com.couchbase.connector.cluster.consul;

import com.couchbase.client.core.deps.com.fasterxml.jackson.databind.JsonNode;
import com.couchbase.client.dcp.core.utils.DefaultObjectMapper;
import com.couchbase.connector.cluster.Membership;
import com.couchbase.connector.cluster.consul.ConsulContext;
import com.couchbase.connector.cluster.consul.LeaderEvent;
import com.couchbase.connector.cluster.consul.ReactorHelper;
import com.couchbase.connector.cluster.consul.WorkerService;
import com.couchbase.connector.cluster.consul.rpc.Broadcaster;
import com.couchbase.connector.cluster.consul.rpc.RpcEndpoint;
import com.couchbase.connector.cluster.consul.rpc.RpcResult;
import com.couchbase.connector.config.es.ConnectorConfig;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import java.io.Closeable;
import java.io.IOException;
import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.Disposable;

public class LeaderTask {
    private static final Logger LOGGER = LoggerFactory.getLogger(LeaderTask.class);
    private static final Duration quietPeriodAfterFailedShutdownRequest = Duration.ofSeconds(30L);
    private final ConsulContext ctx;
    private volatile boolean done;
    private volatile Thread thread;
    private final Broadcaster broadcaster = new Broadcaster();

    public LeaderTask(ConsulContext consulContext) {
        this.ctx = Objects.requireNonNull(consulContext);
    }

    public LeaderTask start() {
        Preconditions.checkState((this.thread == null ? 1 : 0) != 0, (Object)"Already started.");
        this.thread = new Thread(this::doRun);
        this.thread.start();
        return this;
    }

    public void stop() {
        this.done = true;
        this.broadcaster.close();
        if (this.thread != null) {
            this.thread.interrupt();
        }
        this.thread = null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void doRun() {
        LOGGER.info("Leader thread started.");
        boolean hasSeenConfig = false;
        boolean hasSeenClusterMembership = false;
        boolean paused = false;
        LinkedBlockingQueue leaderEvents = new LinkedBlockingQueue();
        try {
            Closeable configWatch = ReactorHelper.asCloseable(this.subscribeConfig(leaderEvents::add));
            try {
                Closeable controlWatch = ReactorHelper.asCloseable(this.subscribeControl(leaderEvents::add));
                try {
                    Closeable membershipWatch = ReactorHelper.asCloseable(this.subscribeMembershipEvents(leaderEvents::add));
                    try {
                        block26: while (true) {
                            this.throwIfDone();
                            LeaderEvent event = (LeaderEvent)((Object)leaderEvents.take());
                            LOGGER.info("Got leadership event: {}", (Object)event);
                            switch (event) {
                                case MEMBERSHIP_CHANGE: {
                                    hasSeenClusterMembership = true;
                                    break;
                                }
                                case CONFIG_CHANGE: {
                                    hasSeenConfig = true;
                                    break;
                                }
                                case PAUSE: {
                                    LOGGER.info("Pausing connector activity.");
                                    paused = true;
                                    this.stopStreaming();
                                    break;
                                }
                                case RESUME: {
                                    if (!paused) {
                                        LOGGER.debug("Ignoring redundant resume signal.");
                                        continue block26;
                                    }
                                    LOGGER.info("Resuming connector activity.");
                                    paused = false;
                                    break;
                                }
                                case FATAL_ERROR: {
                                    throw new RuntimeException("Fatal error in leader task");
                                }
                            }
                            if (hasSeenClusterMembership && hasSeenConfig && !paused) {
                                LOGGER.info("Rebalance triggered by {}", (Object)event);
                                this.rebalance();
                                continue;
                            }
                            if (!hasSeenClusterMembership) {
                                LOGGER.info("Waiting for initial cluster membership event before streaming can start.");
                            }
                            if (!hasSeenConfig) {
                                LOGGER.info("Waiting for connector configuration document to exist before streaming can start.");
                            }
                            if (!paused) continue;
                            LOGGER.info("Connector is paused; waiting for 'resume' control signal before streaming can start.");
                        }
                    }
                    catch (Throwable throwable) {
                        if (membershipWatch != null) {
                            try {
                                membershipWatch.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                }
                catch (Throwable throwable) {
                    if (controlWatch != null) {
                        try {
                            controlWatch.close();
                        }
                        catch (Throwable throwable3) {
                            throwable.addSuppressed(throwable3);
                        }
                    }
                    throw throwable;
                }
            }
            catch (Throwable throwable) {
                if (configWatch != null) {
                    try {
                        configWatch.close();
                    }
                    catch (Throwable throwable4) {
                        throwable.addSuppressed(throwable4);
                    }
                }
                throw throwable;
            }
        }
        catch (InterruptedException e) {
            LOGGER.debug("Leader thread interrupted", (Throwable)e);
            LOGGER.info("Leader thread terminated.");
        }
        catch (Throwable t) {
            try {
                LOGGER.error("panic: Leader task failed", t);
                System.exit(1);
            }
            catch (Throwable throwable) {
                throw throwable;
            }
            finally {
                LOGGER.info("Leader thread terminated.");
            }
        }
    }

    private static JsonNode readTreeOrElseEmptyObject(String s) {
        try {
            return DefaultObjectMapper.readTree((String)(Strings.isNullOrEmpty((String)s) ? "{}" : s));
        }
        catch (IOException e) {
            throw new IllegalArgumentException("Failed to parse JSON", e);
        }
    }

    private void stopStreaming() throws InterruptedException {
        int attempt = 1;
        while (true) {
            this.throwIfDone();
            List<RpcEndpoint> endpoints = this.ctx.rpcEndpoints();
            Map<RpcEndpoint, RpcResult<Void>> stopResults = this.broadcaster.broadcast("stop", endpoints, WorkerService.class, WorkerService::stopStreaming);
            if (stopResults.entrySet().stream().noneMatch(e -> ((RpcResult)e.getValue()).isFailed())) {
                if (attempt != 1) {
                    LOGGER.warn("Multiple attempts were required to quiesce the cluster. Sleeping for an additional {} to allow unreachable nodes to terminate.", (Object)quietPeriodAfterFailedShutdownRequest);
                    LeaderTask.sleep(quietPeriodAfterFailedShutdownRequest);
                }
                LOGGER.info("Cluster quiesced.");
                return;
            }
            LOGGER.warn("Attempt #{} to quiesce the cluster failed. Will retry.", (Object)attempt);
            ++attempt;
            TimeUnit.SECONDS.sleep(5L);
        }
    }

    private static void sleep(Duration d) throws InterruptedException {
        TimeUnit.MILLISECONDS.sleep(Math.max(1L, d.toMillis()));
    }

    public List<RpcEndpoint> awaitReadyEndpoints() throws InterruptedException {
        while (true) {
            this.throwIfDone();
            List<RpcEndpoint> allEndpoints = this.ctx.rpcEndpoints();
            List<RpcEndpoint> readyEndpoints = allEndpoints.stream().filter(rpcEndpoint -> {
                try {
                    rpcEndpoint.service(WorkerService.class).ready();
                    return true;
                }
                catch (Throwable t) {
                    LOGGER.warn("Endpoint {} is not ready; excluding it from rebalance.", rpcEndpoint, (Object)t);
                    return false;
                }
            }).collect(Collectors.toList());
            if (!readyEndpoints.isEmpty()) {
                return readyEndpoints;
            }
            TimeUnit.SECONDS.sleep(5L);
        }
    }

    private void rebalance() throws InterruptedException {
        String config = this.ctx.readConfig();
        ConnectorConfig.from(config);
        block2: while (true) {
            LOGGER.info("Rebalancing the cluster");
            this.stopStreaming();
            List<RpcEndpoint> endpoints = this.awaitReadyEndpoints();
            for (int i = 0; i < endpoints.size(); ++i) {
                this.throwIfDone();
                int memberNumber = i + 1;
                int clusterSize = endpoints.size();
                Membership membership = Membership.of(memberNumber, clusterSize);
                RpcEndpoint endpoint = endpoints.get(i);
                LOGGER.info("Assigning group membership {} to endpoint {}", (Object)membership, (Object)endpoint);
                try {
                    endpoint.service(WorkerService.class).startStreaming(membership, config);
                    continue;
                }
                catch (Throwable t) {
                    LOGGER.warn("Failed to assign group membership {} to endpoint {}", new Object[]{membership, endpoint, t});
                    TimeUnit.SECONDS.sleep(3L);
                    continue block2;
                }
            }
            break;
        }
    }

    private void throwIfDone() throws InterruptedException {
        if (this.done) {
            throw new InterruptedException("Leader termination requested.");
        }
    }

    private Disposable subscribeConfig(Consumer<LeaderEvent> eventSink) {
        return this.ctx.watchConfig().doOnNext(e -> {
            if (e.isPresent()) {
                eventSink.accept(LeaderEvent.CONFIG_CHANGE);
            }
        }).doOnError(e -> {
            LOGGER.error("panic: Config change watcher failed.", e);
            eventSink.accept(LeaderEvent.FATAL_ERROR);
        }).subscribe();
    }

    private Disposable subscribeControl(Consumer<LeaderEvent> eventSink) {
        return this.ctx.watchControl().doOnNext(e -> {
            LOGGER.debug("Got control document: {}", e);
            JsonNode control = LeaderTask.readTreeOrElseEmptyObject(e.orElse(""));
            if (control.path("paused").asBoolean(false)) {
                eventSink.accept(LeaderEvent.PAUSE);
            } else {
                eventSink.accept(LeaderEvent.RESUME);
            }
        }).doOnError(e -> {
            LOGGER.error("panic: Control change watcher failed.", e);
            eventSink.accept(LeaderEvent.FATAL_ERROR);
        }).subscribe();
    }

    private Disposable subscribeMembershipEvents(Consumer<LeaderEvent> eventSink) {
        return this.ctx.watchServiceHealth(Duration.ofSeconds(5L)).doOnNext(membership -> eventSink.accept(LeaderEvent.MEMBERSHIP_CHANGE)).doOnError(e -> {
            LOGGER.error("panic: Service health watcher failed.", e);
            eventSink.accept(LeaderEvent.FATAL_ERROR);
        }).subscribe();
    }
}

