/*
 * Decompiled with CFR 0.152.
 */
package com.couchbase.connector.dcp;

import com.couchbase.connector.dcp.Checkpoint;
import com.couchbase.connector.dcp.CheckpointDao;
import com.couchbase.connector.elasticsearch.Metrics;
import com.google.common.base.Stopwatch;
import io.micrometer.core.instrument.Counter;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReferenceArray;
import java.util.function.Supplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CheckpointService {
    private static final Logger LOGGER = LoggerFactory.getLogger(CheckpointService.class);
    private volatile AtomicReferenceArray<Checkpoint> positions;
    private final CheckpointDao streamPositionDao;
    private final String bucketUuid;
    private final Counter failures = Metrics.counter("save.state.fail", "Failed to save a replication checkpoint document to Couchbase.");
    private volatile boolean dirty;
    private volatile Supplier<Map<Integer, Long>> highSeqnoProvider = Collections::emptyMap;

    public CheckpointService(String bucketUuid, CheckpointDao streamPositionDao) {
        this.bucketUuid = bucketUuid;
        this.streamPositionDao = Objects.requireNonNull(streamPositionDao);
    }

    public void init(int numPartitions, Supplier<Map<Integer, Long>> highSeqnoProvider) {
        this.highSeqnoProvider = Objects.requireNonNull(highSeqnoProvider);
        this.positions = new AtomicReferenceArray(numPartitions);
    }

    public void set(int vbucket, Checkpoint position) {
        this.setWithoutMarkingDirty(vbucket, position);
        this.dirty = true;
    }

    public void setWithoutMarkingDirty(int vbucket, Checkpoint position) {
        LOGGER.debug("New position for vbucket {} is {}", (Object)vbucket, (Object)position);
        this.positions.set(vbucket, position);
    }

    public synchronized Map<Integer, Checkpoint> load(Set<Integer> vbuckets) {
        Map<Integer, Checkpoint> result = this.streamPositionDao.loadOrDefaultToZero(this.bucketUuid, vbuckets);
        LOGGER.debug("Loaded checkpoints: {}", result);
        for (Map.Entry<Integer, Checkpoint> entry : result.entrySet()) {
            int partition = entry.getKey();
            this.setWithoutMarkingDirty(partition, entry.getValue());
        }
        this.registerBacklogMetrics();
        return result;
    }

    public synchronized void save() {
        if (!this.dirty) {
            LOGGER.debug("Connector state unchanged since last save.");
            return;
        }
        try {
            HashMap<Integer, Checkpoint> partitionToPosition = new HashMap<Integer, Checkpoint>();
            for (int i = 0; i < this.positions.length(); ++i) {
                partitionToPosition.put(i, this.positions.get(i));
            }
            this.streamPositionDao.save(this.bucketUuid, partitionToPosition);
            this.dirty = false;
            LOGGER.info("Saved connector state.");
        }
        catch (Exception t) {
            LOGGER.warn("Failed to save connector state.", (Throwable)t);
            this.failures.increment();
        }
    }

    private void registerBacklogMetrics() {
        Metrics.cachedGauge("backlog", "Estimated Couchbase changes yet to be processed by this node.", this, CheckpointService::getLocalBacklog);
    }

    private long getLocalBacklog() {
        Stopwatch timer = Stopwatch.createStarted();
        Map<Integer, Long> highSeqnos = this.highSeqnoProvider.get();
        timer.stop();
        LOGGER.info("Getting current seqnos took {}", (Object)timer);
        long result = 0L;
        for (Map.Entry<Integer, Long> entry : highSeqnos.entrySet()) {
            int partition = entry.getKey();
            long seqno = entry.getValue();
            Checkpoint checkpoint = this.positions.get(partition);
            if (checkpoint == null) {
                LOGGER.warn("Can't calculate local backlog for partition {}; no checkpoint available (yet?)", (Object)partition);
                continue;
            }
            long backlogForPartition = Math.max(0L, seqno - checkpoint.getSeqno());
            LOGGER.debug("Local backlog for partition {}: {} (connector: {} server: {})", new Object[]{partition, backlogForPartition, checkpoint.getSeqno(), seqno});
            try {
                result = Math.addExact(result, backlogForPartition);
            }
            catch (ArithmeticException e) {
                result = Long.MAX_VALUE;
            }
        }
        return result;
    }
}

