/*
 * Decompiled with CFR 0.152.
 */
package com.couchbase.connector.elasticsearch.sink;

import com.couchbase.client.core.logging.RedactableArgument;
import com.couchbase.connector.config.StorageSize;
import com.couchbase.connector.config.es.BulkRequestConfig;
import com.couchbase.connector.dcp.Checkpoint;
import com.couchbase.connector.dcp.CheckpointService;
import com.couchbase.connector.dcp.DcpHelper;
import com.couchbase.connector.dcp.Event;
import com.couchbase.connector.elasticsearch.DocumentLifecycle;
import com.couchbase.connector.elasticsearch.ErrorListener;
import com.couchbase.connector.elasticsearch.Metrics;
import com.couchbase.connector.elasticsearch.io.BackoffPolicy;
import com.couchbase.connector.elasticsearch.io.BackoffPolicyBuilder;
import com.couchbase.connector.elasticsearch.io.RequestFactory;
import com.couchbase.connector.elasticsearch.sink.Operation;
import com.couchbase.connector.elasticsearch.sink.RejectOperation;
import com.couchbase.connector.elasticsearch.sink.RetryReporter;
import com.couchbase.connector.elasticsearch.sink.SinkBulkResponse;
import com.couchbase.connector.elasticsearch.sink.SinkBulkResponseItem;
import com.couchbase.connector.elasticsearch.sink.SinkErrorCause;
import com.couchbase.connector.elasticsearch.sink.SinkOps;
import com.couchbase.connector.util.ThrowableHelper;
import java.io.Closeable;
import java.io.IOException;
import java.net.ConnectException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import javax.annotation.concurrent.GuardedBy;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SinkWriter
implements Closeable {
    private static final Logger LOGGER = LoggerFactory.getLogger(SinkWriter.class);
    private final SinkOps client;
    private final RequestFactory requestFactory;
    private final CheckpointService checkpointService;
    private final ErrorListener errorListener = ErrorListener.NOOP;
    private final long bufferBytesThreshold;
    private final int bufferActionsThreshold;
    private final Duration bulkRequestTimeout;
    private static final Duration INITIAL_RETRY_DELAY = Duration.ofMillis(50L);
    private static final Duration MAX_RETRY_DELAY = Duration.ofMinutes(5L);
    private final BackoffPolicy backoffPolicy = BackoffPolicyBuilder.truncatedExponentialBackoff(INITIAL_RETRY_DELAY, MAX_RETRY_DELAY).fullJitter().build();
    @GuardedBy(value="this")
    private boolean requestInProgress;
    @GuardedBy(value="this")
    private long requestStartNanos;
    private final LinkedHashMap<String, Operation> buffer = new LinkedHashMap();
    private int bufferBytes;
    private final Map<Integer, Checkpoint> ignoreBuffer = new HashMap<Integer, Checkpoint>();

    public SinkWriter(SinkOps client, CheckpointService checkpointService, RequestFactory requestFactory, BulkRequestConfig bulkConfig) {
        this.client = Objects.requireNonNull(client);
        this.checkpointService = Objects.requireNonNull(checkpointService);
        this.requestFactory = Objects.requireNonNull(requestFactory);
        this.bufferActionsThreshold = bulkConfig.maxActions();
        this.bufferBytesThreshold = bulkConfig.maxBytes().getBytes();
        this.bulkRequestTimeout = Objects.requireNonNull(bulkConfig.timeout());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void write(Event event) throws InterruptedException {
        Operation request = this.requestFactory.newDocWriteRequest(event);
        if (request == null) {
            try {
                if (LOGGER.isTraceEnabled()) {
                    LOGGER.trace("Skipping event, no matching type: {}", (Object)RedactableArgument.redactUser((Object)event));
                }
                if (this.buffer.isEmpty()) {
                    Checkpoint checkpoint = event.getCheckpoint();
                    if (DcpHelper.isMetadata(event)) {
                        LOGGER.debug("Ignoring metadata, not updating checkpoint for {}", (Object)event);
                        this.checkpointService.setWithoutMarkingDirty(event.getVbucket(), event.getCheckpoint());
                    } else {
                        LOGGER.debug("Ignoring event, immediately updating checkpoint for {}", (Object)event);
                        this.checkpointService.set(event.getVbucket(), checkpoint);
                    }
                } else {
                    this.ignoreBuffer.put(event.getVbucket(), event.getCheckpoint());
                }
                return;
            }
            finally {
                event.release();
            }
        }
        this.bufferBytes += request.estimatedSizeInBytes();
        Operation evicted = this.buffer.put(event.getKey() + "\u0000" + request.getIndex(), request);
        if (evicted != null) {
            String evictedQualifiedDocId;
            String qualifiedDocId = event.getKey(true);
            if (!qualifiedDocId.equals(evictedQualifiedDocId = evicted.getEvent().getKey(true))) {
                LOGGER.warn("DOCUMENT ID COLLISION DETECTED: Documents '{}' and '{}' are from different collections but have the same destination index '{}'.", new Object[]{qualifiedDocId, evictedQualifiedDocId, request.getIndex()});
            }
            DocumentLifecycle.logSkippedBecauseNewerVersionReceived(evicted.getEvent(), event.getTracingToken());
            this.bufferBytes -= evicted.estimatedSizeInBytes();
            evicted.getEvent().release();
        }
        if (this.bufferIsFull()) {
            this.flush();
        }
    }

    private Checkpoint adjustForIgnoredEvents(int vbucket, Checkpoint checkpoint) {
        Checkpoint ignored = this.ignoreBuffer.remove(vbucket);
        if (ignored == null) {
            return checkpoint;
        }
        if (ignored.getVbuuid() != checkpoint.getVbuuid()) {
            LOGGER.debug("vbuuid of ignored event does not match last written event (rollback?); will disregard ignored event when updating checkpoint.");
            return checkpoint;
        }
        if (Long.compareUnsigned(ignored.getSeqno(), checkpoint.getSeqno()) > 0) {
            LOGGER.debug("Adjusting vbucket {} checkpoint {} for ignored events -> {}", new Object[]{vbucket, checkpoint, ignored});
            return ignored;
        }
        return checkpoint;
    }

    private boolean bufferIsFull() {
        return this.buffer.size() >= this.bufferActionsThreshold || (long)this.bufferBytes >= this.bufferBytesThreshold;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void flush() throws InterruptedException {
        if (this.buffer.isEmpty()) {
            return;
        }
        try {
            SinkWriter sinkWriter = this;
            synchronized (sinkWriter) {
                this.requestInProgress = true;
                this.requestStartNanos = System.nanoTime();
            }
            int totalActionCount = this.buffer.size();
            int totalEstimatedBytes = this.bufferBytes;
            LOGGER.debug("Starting bulk request: {} actions for ~{} bytes", (Object)totalActionCount, (Object)totalEstimatedBytes);
            long startNanos = System.nanoTime();
            ArrayList<Operation> requests = new ArrayList<Operation>(this.buffer.values());
            this.clearBuffer();
            Iterator waitIntervals = this.backoffPolicy.iterator();
            Map<Integer, Operation> vbucketToLastEvent = SinkWriter.lenientIndex(r -> r.getEvent().getVbucket(), requests);
            int attemptCounter = 1;
            Duration indexingTook = Duration.ZERO;
            long totalRetryDelayMillis = 0L;
            while (true) {
                if (Thread.interrupted()) {
                    requests.forEach(r -> r.getEvent().release());
                    Thread.currentThread().interrupt();
                    return;
                }
                DocumentLifecycle.logEsWriteStarted(requests, attemptCounter);
                if (attemptCounter == 1) {
                    LOGGER.debug("Bulk request attempt #{}", (Object)attemptCounter++);
                } else {
                    LOGGER.info("Bulk request attempt #{}", (Object)attemptCounter++);
                }
                ArrayList<Operation> requestsToRetry = new ArrayList<Operation>(0);
                RetryReporter retryReporter = RetryReporter.forLogger(LOGGER);
                try {
                    SinkBulkResponse bulkResponse = this.client.bulk(requests);
                    long l = System.nanoTime();
                    List<SinkBulkResponseItem> responses = bulkResponse.items();
                    indexingTook = indexingTook.plus(bulkResponse.ingestTook());
                    for (int i = 0; i < responses.size(); ++i) {
                        SinkBulkResponseItem response = responses.get(i);
                        SinkErrorCause failure = response.error();
                        Operation request = (Operation)requests.get(i);
                        Event e = request.getEvent();
                        if (failure == null) {
                            SinkWriter.updateLatencyMetrics(e, l);
                            DocumentLifecycle.logEsWriteSucceeded(request);
                            e.release();
                            continue;
                        }
                        if (SinkWriter.isRetryable(response)) {
                            retryReporter.add(e, response);
                            requestsToRetry.add(request);
                            DocumentLifecycle.logEsWriteFailedWillRetry(request);
                            continue;
                        }
                        if (request instanceof RejectOperation) {
                            LOGGER.error("Failed to index rejection document for event {}; status code: {} {}", new Object[]{RedactableArgument.redactUser((Object)e), response.status(), failure.reason()});
                            Metrics.rejectionLogFailureCounter().increment();
                            SinkWriter.updateLatencyMetrics(e, l);
                            e.release();
                        } else {
                            LOGGER.warn("Permanent failure to index event {}; status code: {} {}", new Object[]{RedactableArgument.redactUser((Object)e), response.status(), failure.reason()});
                            Metrics.rejectionCounter().increment();
                            DocumentLifecycle.logEsWriteRejected(request, response.status(), failure.toString());
                            RejectOperation rejectionLogRequest = this.requestFactory.newRejectionLogRequest(request, response);
                            if (rejectionLogRequest != null) {
                                requestsToRetry.add(rejectionLogRequest);
                            }
                        }
                        SinkWriter.runQuietly("error listener", () -> this.errorListener.onFailedIndexResponse(e, response));
                    }
                    Metrics.indexingRetryCounter().increment((double)requestsToRetry.size());
                    requests = requestsToRetry;
                }
                catch (IOException e) {
                    if (ThrowableHelper.hasCause(e, ConnectException.class, new Class[0])) {
                        LOGGER.debug("Elasticsearch connect exception", (Throwable)e);
                        LOGGER.warn("Bulk request failed; could not connect to Elasticsearch.");
                    } else {
                        LOGGER.warn("Bulk request failed", (Throwable)e);
                    }
                }
                catch (RuntimeException e) {
                    requests.forEach(r -> r.getEvent().release());
                    ThrowableHelper.propagateCauseIfPossible(e, InterruptedException.class);
                    throw e;
                }
                if (requests.isEmpty()) {
                    for (Map.Entry<Integer, Operation> entry : vbucketToLastEvent.entrySet()) {
                        int vbucket = entry.getKey();
                        Checkpoint checkpoint = entry.getValue().getEvent().getCheckpoint();
                        checkpoint = this.adjustForIgnoredEvents(vbucket, checkpoint);
                        this.checkpointService.set(entry.getKey(), checkpoint);
                    }
                    for (Map.Entry<Integer, Object> entry : this.ignoreBuffer.entrySet()) {
                        this.checkpointService.set(entry.getKey(), (Checkpoint)entry.getValue());
                    }
                    Metrics.bytesCounter().increment((double)totalEstimatedBytes);
                    Metrics.indexTimePerDocument().record(indexingTook.dividedBy(totalActionCount));
                    if (totalRetryDelayMillis != 0L) {
                        Metrics.retryDelayTimer().record(totalRetryDelayMillis, TimeUnit.MILLISECONDS);
                    }
                    if (LOGGER.isInfoEnabled()) {
                        long elapsedMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startNanos);
                        StorageSize prettySize = StorageSize.ofBytes(totalEstimatedBytes);
                        LOGGER.info("Wrote {} actions ~{} in {} ms", new Object[]{totalActionCount, prettySize, elapsedMillis});
                    }
                    return;
                }
                retryReporter.report();
                Metrics.bulkRetriesCounter().increment();
                Duration retryDelay = (Duration)waitIntervals.next();
                LOGGER.info("Retrying bulk request in {}", (Object)retryDelay);
                TimeUnit.MILLISECONDS.sleep(retryDelay.toMillis());
                totalRetryDelayMillis += retryDelay.toMillis();
            }
        }
        finally {
            SinkWriter sinkWriter = this;
            synchronized (sinkWriter) {
                this.requestInProgress = false;
            }
        }
    }

    public synchronized long getCurrentRequestNanos() {
        return this.requestInProgress ? System.nanoTime() - this.requestStartNanos : 0L;
    }

    private static void updateLatencyMetrics(Event e, long nowNanos) {
        long elapsedNanos = nowNanos - e.getReceivedNanos();
        Metrics.latencyTimer().record(elapsedNanos, TimeUnit.NANOSECONDS);
    }

    private void clearBuffer() {
        this.buffer.clear();
        this.bufferBytes = 0;
    }

    private static boolean isRetryable(SinkBulkResponseItem f) {
        if (f.error() == null) {
            throw new IllegalArgumentException("bulk response item didn't fail");
        }
        switch (f.status()) {
            case 400: 
            case 404: {
                return false;
            }
        }
        return true;
    }

    private static void runQuietly(String description, Runnable r) {
        try {
            r.run();
        }
        catch (Exception e) {
            LOGGER.warn("Exception in {}", (Object)description, (Object)e);
        }
    }

    private static <K, V> Map<K, V> lenientIndex(Function<V, K> keyGenerator, Iterable<V> items) {
        HashMap<K, V> result = new HashMap<K, V>();
        for (V item : items) {
            result.put(keyGenerator.apply(item), item);
        }
        return result;
    }

    @Override
    public void close() {
        this.buffer.values().forEach(e -> e.getEvent().release());
    }
}

